/**
* \file: logger.c
*
* \version: $Id:$
*
* \release: $Name:$
*
* \component: authorization level daemon
*
* \author: Marko Hoyer / ADIT / SWGII / mhoyer@de.adit-jv.com
*
* \copyright (c) 2010, 2011 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
*
***********************************************************************/
#include "util/logger.h"

#include <syslog.h>
#include <stdbool.h>
#include <stdio.h>
#include <string.h>
#include <stdarg.h>
#include <pthread.h>
#include <errno.h>

static pthread_mutex_t logger_mutex = PTHREAD_MUTEX_INITIALIZER;

static bool logger_syslog_active=false;
static bool logger_consolelog_active=false;

static logger_loglevel_t logger_min_level=LOGGER_LEVEL_INFO;

static bool logger_console_enabled=true;

const char *logger_proc_name=NULL;

static void logger_console_log(FILE* stream, const char *level, const char *message, va_list args);

void logger_init(const char *proc_name, logger_loglevel_t min_level, bool console_enabled)
{
	logger_min_level=min_level;
	logger_console_enabled=console_enabled;
	logger_proc_name=proc_name;

	logger_open_syslog();
	logger_open_console_log();
}

void logger_init_console_only(logger_loglevel_t min_level)
{
	logger_min_level=min_level;
	logger_console_enabled=true;
	logger_open_console_log();
}

void logger_deinit(void)
{
	logger_close_syslog();
	logger_close_console_log();
}

error_code_t logger_parse_loglevel(const char *level_str, logger_loglevel_t *result_ptr)
{
	if (strcasecmp(level_str,"none")==0)
		*result_ptr=LOGGER_LEVEL_NONE;
	else if (strcasecmp(level_str,"error")==0)
		*result_ptr=LOGGER_LEVEL_ERROR;
	else if (strcasecmp(level_str,"info")==0)
		*result_ptr=LOGGER_LEVEL_INFO;
	else if (strcasecmp(level_str,"debug")==0)
		*result_ptr=LOGGER_LEVEL_DEBUG;
	else
		return RESULT_INVALID_ARGS;

	return RESULT_OK;
}


void logger_set_log_level(logger_loglevel_t log_level)
{
	logger_min_level=log_level;
}

void logger_set_console_enabled(bool enabled)
{
	logger_console_enabled=enabled;
}


void logger_open_syslog(void)
{
	if (!logger_syslog_active)
	{
		openlog(logger_proc_name, 0, LOG_DAEMON);
		logger_syslog_active=true;
	}
}

void logger_close_syslog(void)
{
	if (logger_syslog_active)
	{
		closelog();
		logger_syslog_active=true;
	}
}

void logger_close_console_log(void)
{
	logger_consolelog_active=false;
}

void logger_open_console_log(void)
{
	logger_consolelog_active=logger_console_enabled;
}

void logger_log_error(const char *message, ...)
{
	va_list args;

	if (logger_min_level==LOGGER_LEVEL_NONE)
		return;

	pthread_mutex_lock(&logger_mutex);
	memset(&args,0,sizeof(va_list));

	if (logger_consolelog_active)
	{
		va_start (args, message);
		logger_console_log(stderr,"ERROR",message,args);
		va_end(args);
	}


	if (logger_syslog_active)
	{
		va_start (args, message);
		vsyslog(LOG_ERR,message,args);
		va_end(args);
	}
	pthread_mutex_unlock(&logger_mutex);
}

void logger_log_info(const char *message, ...)
{
	va_list args;

	if (logger_min_level<LOGGER_LEVEL_INFO)
		return;

	pthread_mutex_lock(&logger_mutex);
	memset(&args,0,sizeof(va_list));

	if (logger_consolelog_active)
	{
		va_start (args, message);
		logger_console_log(stderr,"INFO",message,args);
		va_end(args);
	}

	if (logger_syslog_active)
	{
		va_start (args, message);
		vsyslog(LOG_INFO,message,args);
		va_end(args);
	}
	pthread_mutex_unlock(&logger_mutex);
}

void logger_log_status(const char *message, ...)
{
	va_list args;
	pthread_mutex_lock(&logger_mutex);
	memset(&args,0,sizeof(va_list));

	if (logger_consolelog_active)
	{
		va_start (args, message);
		logger_console_log(stderr,NULL,message,args);
		va_end(args);
	}

	if (logger_syslog_active)
	{
		va_start (args, message);
		vsyslog(LOG_INFO,message,args);
		va_end(args);
	}
	pthread_mutex_unlock(&logger_mutex);
}

void logger_log_debug(const char *message, ...)
{
	va_list args;
	if (logger_min_level<LOGGER_LEVEL_DEBUG)
		return;

	pthread_mutex_lock(&logger_mutex);
	memset(&args,0,sizeof(va_list));

	if (logger_consolelog_active)
	{
		va_start(args, message);
		logger_console_log(stderr,"DEBUG",message,args);
		va_end(args);
	}

	if (logger_syslog_active)
	{
		va_start(args, message);
		vsyslog(LOG_DEBUG,message,args);
		va_end(args);
	}
	pthread_mutex_unlock(&logger_mutex);
}

void logger_log_errmem(const char *message, ...)
{
	va_list args;
	FILE *errmem_fd;
	memset(&args,0,sizeof(va_list));

	errmem_fd=fopen(ERRMEM_DEVICE,"w");

	if (errmem_fd!=NULL)
	{
		pthread_mutex_lock(&logger_mutex);
		va_start(args, message);
		vfprintf(errmem_fd, message, args);
		fprintf(errmem_fd,"\n");
		va_end(args);
		fclose(errmem_fd);
		pthread_mutex_unlock(&logger_mutex);
	}
	else
		logger_log_error("Unable to log into error memory. Message: %s", strerror(errno));
}

static void logger_console_log(FILE* stream, const char *level, const char *message, va_list args)
{
	if (level!=NULL)
		fprintf(stream,"%s: ",level);
	vfprintf(stream, message, args);
	fprintf(stream, "\n");
}
